// $Id: CVstWindow.cpp,v 1.11 2007/03/03 02:59:45 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 *
 * Please note that VST is copyright Steinberg Media GmBh. No challenge is made to any of their trademarks
 * To use this file, you require a copy of the VST SDK, available from www.steinberg.net for free
 */

#include "CVstWindow.hpp"
using Exponent::Vst::CVstWindow;

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CVstWindow, CWindow);

//	===========================================================================
CVstWindow::CVstWindow(AudioEffect *theEffect, const CRect &windowArea, CWindowAttributes *attributes) 
		  : AEffEditor (theEffect)
		  , CWindow(windowArea)
#ifndef WIN32
		  , m_idleSection(NULL)
		  , m_userPane(NULL)
		  , m_contentPane(NULL)
		  , m_windowTimer(NULL)
#endif
{
	EXPONENT_CLASS_CONSTRUCTION(CVstWindow);

	theEffect->setEditor(this);

	// Setup the attributes
	*m_attributes = *attributes;
#ifdef WIN32
	m_attributes->initialise(WS_CHILD | WS_VISIBLE, WS_EX_ACCEPTFILES);
#else
	// On the mac, we dont case what the attributes are because we are just going to register with the window the host passes us
	m_updateAreaIsValid = false;
	
	m_idleSection = new CCriticalSection;
	m_idleSection->referenced();
#endif

	NULL_POINTER(m_theVSTParentWindow);

	// Notify the root that it is VST aware
	m_theControlRoot->isVSTAttatched();

	// No controls have been set
	m_setFromEditor = false;
	m_blockUpdates  = false;
}

//	===========================================================================
CVstWindow::~CVstWindow()
{
	EXPONENT_CLASS_DESTRUCTION(CVstWindow);
	FREE_POINTER(m_theVSTParentWindow);
#ifndef WIN32
	FORGET_COUNTED_OBJECT(m_idleSection);
#endif
}

//	===========================================================================
#ifdef VST_2_4_EXTENSIONS
bool CVstWindow::getRect(ERect **rect)
#else
long CVstWindow::getRect(ERect **rect)
#endif
{
	m_myRect.left	= (short)0;
	m_myRect.top    = (short)0;
	m_myRect.right  = (short)m_windowDimension.getWidth();
	m_myRect.bottom = (short)m_windowDimension.getHeight();
	*rect = &m_myRect;
	#ifdef VST_2_4_EXTENSIONS
	return true;
	#else
	return 1;
	#endif
}

//	===========================================================================
void CVstWindow::update()
{
#ifdef VST_2_4_EXTENSIONS
	this->redrawWindow();
#else
	if (updateFlag == 1)
	{
		updateFlag = 0;
		this->redrawWindow();
	}
#endif
}

//	===========================================================================
void CVstWindow::redrawWindow(const CRect &area)
{
	if (!m_blockUpdates)
	{
		CWindow::redrawWindow(area);
	}
}

//	===========================================================================
void CVstWindow::redrawWindow()
{
	if (!m_blockUpdates)
	{
		CWindow::redrawWindow();
	}
}

//	===========================================================================
#ifdef VST_2_4_EXTENSIONS
bool CVstWindow::open(void *windowHandle)
#else
long CVstWindow::open(void *windowHandle)
#endif
{
	this->createVSTWindow(windowHandle);
	this->startVSTWindow();
	#ifdef VST_2_4_EXTENSIONS
		return true;
	#else
		return 1;
	#endif
}

//	===========================================================================
void CVstWindow::close()
{
	this->destroyVSTWindow();
}

//	===========================================================================
void CVstWindow::startVSTWindow()
{
#ifdef WIN32
	this->initialiseWindow(m_attributes, true);
#else
	//this->openWindow();
	//this->updateWindow();
#endif
}

//	===========================================================================
void CVstWindow::createVSTWindow(void *windowHandle)
{
	// Check we have a valid handle
	if (windowHandle == NULL)
	{
		throw CException("Parent window handle is NULL", "CVSTPanel::open(void *)");
	}

	// Satisfy parent
	AEffEditor::open(windowHandle);

#ifdef WIN32
	// On WIN32 machines, we are a child window of the main window,
	// therefore all we have to do is to store the attributes
	// and the window handle as the parent and then call
	// initialise in an overridden version
	// Now you are safe in an overrided class to add your controls and call startVSTWindow

	FREE_POINTER(m_theVSTParentWindow);
	m_theVSTParentWindow = new CParentWindow(windowHandle);
	this->setParentWindow(m_theVSTParentWindow);
#else

	// On MAC machines, we are going to register as an event handler for this window.
	// Here is gets a bit tricky, because we cannot just create a child window and have
	// that handle everything, we have to do it all again :(

	// The events we respond to
	const static EventTypeSpec vstEventSpec[] =
	{
		{ kEventClassMouse, kEventMouseDown					},
		{ kEventClassMouse, kEventMouseUp					},
		{ kEventClassMouse, kEventMouseMoved				},
		{ kEventClassMouse, kEventMouseDragged				},
		{ kEventClassMouse, kEventMouseWheelMoved			},

		{ kEventClassKeyboard, kEventRawKeyDown				},
		{ kEventClassKeyboard, kEventRawKeyUp				},
		{ kEventClassKeyboard, kEventRawKeyRepeat			},

		{ kEventClassWindow, kEventWindowDeactivated		},
		{ kEventClassWindow, kEventWindowFocusAcquired		},
		{ kEventClassWindow, kEventWindowFocusRelinquish	},
	};

	// The window parent
	m_windowHandle->m_windowHandle = (WindowRef)windowHandle;
	
	//CTrace::trace("Checking for compositing window attributes");

	if (CWindowTools::isWindowComposited(m_windowHandle))
	{
		//CTrace::trace("Compositing window found");
		
		Rect rect;
		rect.top    = 0;
		rect.left   = 0;
		rect.bottom = m_windowDimension.getHeight();
		rect.right  = m_windowDimension.getWidth();
		
		// Find the content pane
		if (HIViewFindByID(HIViewGetRoot(m_windowHandle->m_windowHandle), kHIViewWindowContentID, &m_contentPane) == noErr)
		{
			// Create the user pane that holds our controls
			if (CreateUserPaneControl(m_windowHandle->m_windowHandle, &rect, 0, &m_userPane) == noErr)
			{
				// We register for two basic events
				const static EventTypeSpec userPaneEvents[] =
				{
					{ kEventClassControl, kEventControlBoundsChanged },
					{ kEventClassControl, kEventControlDraw },
				};
				
				// Now add our sub view to the parent view
				if (HIViewAddSubview(m_contentPane, m_userPane) == noErr)
				{
					// Store this for the updates
					this->setCompositingControl(m_userPane);
				
					// Now install the event handlers
					InstallControlEventHandler(m_userPane,    NewEventHandlerUPP(HIViewControlProc), GetEventTypeCount(userPaneEvents),   userPaneEvents, this, NULL);
					InstallControlEventHandler(m_contentPane, NewEventHandlerUPP(HIViewControlProc), 1 /* No support for drawing this */, userPaneEvents, this, NULL);
				}			
			}
		}
	}
	else
	{
		#ifdef VST_2_4_EXTENSIONS
			m_windowTimer = new CVstWindowTimer(this);
			this->addTimedObject(m_windowTimer);
			this->startTimer(m_windowTimer->getTimerId(), 25);
			m_updateAreaIsValid = true;
			m_redrawUpdateArea.setRect(0, 0, m_windowDimension.getWidth(), m_windowDimension.getHeight());
		#endif
	}

	// Install the event handlers
	InstallEventHandler(GetWindowEventTarget((WindowRef)windowHandle), NewEventHandlerUPP(VSTWindowProc), GetEventTypeCount(vstEventSpec), vstEventSpec, this, NULL);
	InstallReceiveHandler(NewDragReceiveHandlerUPP(handleDropMessages), (WindowRef)windowHandle, this);

	// initialised and we are done..
	m_windowInitialised = true;
#endif
}

//	===========================================================================
void CVstWindow::destroyVSTWindow()
{
	FREE_POINTER(m_theVSTParentWindow);
#ifdef WIN32
	this->uninitialiseWindow();
	this->destroyWindow();
#else
	if (m_userPane)
	{
		HIViewRemoveFromSuperview (m_userPane);
	}
	#ifdef VST_2_4_EXTENSIONS
	if (m_windowTimer)
	{
		this->stopTimer(m_windowTimer->getTimerId());
		FREE_POINTER(m_windowTimer);
	}
	#endif
	this->unregisterAllChildWindows();
	this->clearTimedObjects();
	NULL_POINTER(m_windowHandle->m_windowHandle);
	m_windowInitialised = false;
#endif
}

//	===========================================================================
void CVstWindow::handleActionEvent(const CActionEvent &event)
{
	// Overriding class can handle this if they fancy it!
}

//	===========================================================================
void CVstWindow::setParameter(const long index, const double value)
{
	// Overriding class can handle this if they fancy it!
}

//	===========================================================================
bool CVstWindow::handleKeyDown(const CKeyboardEvent &event)
{
	if (m_keyboardListener)
	{
		return m_keyboardListener->handleKeyDown(event);
	}
	return false;
}

//	===========================================================================
bool CVstWindow::handleKeyUp(const CKeyboardEvent &event)
{
	if (m_keyboardListener)
	{
		return m_keyboardListener->handleKeyUp(event);
	}
	return false;
}

#ifndef WIN32

//	===========================================================================
pascal OSStatus CVstWindow::handleVstWindowEvents(EventHandlerCallRef handler, EventRef theEvent)
{
	OSStatus result = eventNotHandledErr;

	UInt32 myEventClass = GetEventClass(theEvent);
	UInt32 myEventKind  = GetEventKind(theEvent);

	// Handle the event
	switch(myEventClass)
	{
		case kEventClassKeyboard:
			switch(myEventKind)
			{
				case kEventRawKeyDown:
					if (m_keyboardListener)
					{
						this->constructKeyboardEvent(theEvent);
						m_keyboardListener->handleKeyDown(m_keyEvent);
						result = noErr;
					}
				break;
				case kEventRawKeyUp:
					if (m_keyboardListener)
					{
						this->constructKeyboardEvent(theEvent);
						m_keyboardListener->handleKeyUp(m_keyEvent);
						result = noErr;
					}
				break;
				case kEventRawKeyRepeat:
					if (m_keyboardListener)
					{
						this->constructKeyboardEvent(theEvent);
						m_keyboardListener->handleKeyDown(m_keyEvent);
						result = noErr;
					}
				break;
			}
		break;


		case kEventClassMouse:
			switch(myEventKind)
			{
				case kEventMouseDown:
					{
						Point point;
						GetEventParameter(theEvent, kEventParamMouseLocation, typeQDPoint, NULL, sizeof(Point), NULL, &point);
						WindowRef windowHandle;
						const WindowPartCode partCode = FindWindow(point, &windowHandle);
						
						// =================================================
						//
						// A Note: Some hosts (ableton live grrrr...) push their plugins
						// in to a kFloatingWindowClass that doesnt properly recieve focus. 
						// This means when we click on our editor from the menus, we dont
						// properly close the main menu window. 
						// This hack below should change that 
						//
						// =================================================
						
						WindowClass windowClass;
						GetWindowClass(windowHandle, &windowClass);
						
						if (windowClass == kFloatingWindowClass)
						{
							if (GetUserFocusWindow() != this->getWindowHandle()->m_windowHandle)
							{
								SetUserFocusWindow(this->getMutableWindowHandle()->m_windowHandle);						
							}
						}

						if (!IsWindowActive(windowHandle))
						{
							return eventNotHandledErr;
						}

						if (partCode == inContent)
						{
							if (m_mouseListener)
							{
								this->constructMousePosition(theEvent);

								if (m_mouseEvent.getMousePosition().getXPosition() >= m_windowDimension.getWidth() ||
								    m_mouseEvent.getMousePosition().getYPosition() >= m_windowDimension.getHeight())
								{
									return eventNotHandledErr;
								}

								// Obtain the button
								UInt16 theButton;
								GetEventParameter(theEvent, kEventParamMouseButton, typeMouseButton, NULL, sizeof(UInt16), NULL, &theButton);

								// Or modifiers & command
								if (theButton == kEventMouseButtonPrimary)
								{
									// Get click count
									UInt32 clickCount;
									GetEventParameter(theEvent, kEventParamClickCount, typeUInt32, NULL, sizeof(UInt32), NULL, &clickCount);

									// Check if single or double click is required
									if (clickCount == 2)
									{
										m_mouseListener->handleDoubleClick(m_mouseEvent);
									}
									else
									{
										if (m_mouseEvent.isAltDown())
										{
											m_leftClick = false;
											m_mouseListener->handleRightButtonDown(m_mouseEvent);
										}
										else
										{
											m_leftClick = true;
											m_mouseListener->handleLeftButtonDown(m_mouseEvent);
										}
									}
								}
								else if (theButton == kEventMouseButtonSecondary)
								{
									m_leftClick = false;
									m_mouseListener->handleRightButtonDown(m_mouseEvent);
								}
								result = noErr;
							}
						}
						else
						{
							return eventNotHandledErr;
						}
					}
				break;
				case kEventMouseUp:
					if (m_mouseListener)
					{
						this->constructMousePosition(theEvent);

						if (m_mouseEvent.getMousePosition().getXPosition() >= m_windowDimension.getWidth() ||
							m_mouseEvent.getMousePosition().getYPosition() >= m_windowDimension.getHeight())
						{
							return eventNotHandledErr;
						}

						UInt16 theButton;
						GetEventParameter(theEvent, kEventParamMouseButton, typeMouseButton, NULL, sizeof(UInt16), NULL, &theButton);

						if (theButton == kEventMouseButtonPrimary || m_leftClick)
						{
							m_mouseListener->handleLeftButtonUp(m_mouseEvent);
						}
						else if (theButton == kEventMouseButtonSecondary)
						{
							m_mouseListener->handleRightButtonUp(m_mouseEvent);
						}
					}
				break;
				case kEventMouseMoved:
					if (m_mouseListener)
					{
						this->constructMousePosition(theEvent);

						if (m_mouseEvent.getMousePosition().getXPosition() >= m_windowDimension.getWidth() ||
							m_mouseEvent.getMousePosition().getYPosition() >= m_windowDimension.getHeight())
						{
							return eventNotHandledErr;
						}

						// Get the position
						Point point;
						GetEventParameter(theEvent, kEventParamMouseLocation, typeQDPoint, NULL, sizeof(Point), NULL, &point);

						// Get the window handle
						WindowRef windowHandle;
						FindWindow(point, &windowHandle);

						// Get the bounds
						Rect windBounds;
						GetWindowBounds(windowHandle, kWindowContentRgn, &windBounds);

						if (PtInRect(point, &windBounds))
						{
							m_mouseListener->handleMouseMovement(m_mouseEvent);
						}
						else
						{
							m_mouseListener->handleMouseLeavingArea(m_mouseEvent);
						}
						result = noErr;
					}
				break;
				case kEventMouseDragged:
					if (m_mouseListener)
					{
						this->constructMousePosition(theEvent);

						if (m_mouseEvent.getMousePosition().getXPosition() >= m_windowDimension.getWidth() ||
							m_mouseEvent.getMousePosition().getYPosition() >= m_windowDimension.getHeight())
						{
							return eventNotHandledErr;
						}
						m_mouseListener->handleMouseMovement(m_mouseEvent);
						result = noErr;
					}
				break;
				case kEventMouseWheelMoved:
					if (m_mouseListener)
					{
						SInt32 delta;
						GetEventParameter(theEvent, kEventParamMouseWheelDelta, typeLongInteger, NULL, sizeof(delta), NULL, &delta);

						this->constructMousePosition(theEvent);
						m_mouseEvent.setWheelMovementAmount((double)delta);

						if (m_mouseEvent.getMousePosition().getXPosition() >= m_windowDimension.getWidth() ||
							m_mouseEvent.getMousePosition().getYPosition() >= m_windowDimension.getHeight())
						{
							return eventNotHandledErr;
						}

						m_mouseListener->handleMouseScroll(m_mouseEvent);
						result = noErr;
					}
				break;
			}
		break;
		case kEventClassWindow:
			switch(myEventKind)
			{
				case kEventWindowActivated:
					this->redrawWindow();
				break;
				case kEventWindowFocusAcquired:
					this->redrawWindow();
					if (m_focusListener)
					{
						m_focusListener->gainedFocus(this);
					}
				break;
				case kEventWindowFocusRelinquish:
					this->redrawWindow();
					if (m_focusListener)
					{
						m_focusListener->lostFocus(this);
					}
				break;
				case kEventWindowDrawContent:
					{
						//CTrace::trace("Window event draw");
						// Get the window handle
						WindowRef windowHandle;
						GetEventParameter(theEvent, kEventParamDirectObject, typeWindowRef, NULL, sizeof(windowHandle), NULL, &windowHandle);

						// Get the current port
						CGrafPtr thePort;
						GetPort(&thePort);

						// Store the update area
						Rect updateArea;
						GetPortBounds(thePort, &updateArea);

						// Setup the qwartz context
						CGContextRef theContext;

						// Start the context drawing
						QDBeginCGContext(thePort, &theContext);

						// Clip to the area
						CGContextSaveGState(theContext);

						// Setup the graphics object
						m_graphics.initialise(m_windowHandle, theContext, m_windowDimension);

						CRect myUpdate;
						myUpdate.setFromRect(updateArea);
						m_graphics.setUpdateArea(myUpdate);
						m_graphics.resetDrawingArea();

						// Draw the images
						m_theControlRoot->drawRootControl(m_graphics);

						// destup the graphics
						m_graphics.resetDrawingArea();
						m_graphics.uninitialise();

						// Restore the clipped rectangle
						CGContextRestoreGState(theContext);

						// Flush the content to the window
						CGContextFlush(theContext);

						// End the use of the context
						QDEndCGContext(thePort, &theContext);

						// And we are done
						result = noErr;
					}
				break;
			}
		break;

	}
	return result;
}

//	===========================================================================
pascal OSStatus CVstWindow::VSTWindowProc(EventHandlerCallRef handler, EventRef theEvent, void *userData)
{
	if (userData)
	{
		CVstWindow *theWindow = (CVstWindow *)userData;
		return theWindow->handleVstWindowEvents(handler, theEvent);
	}
	return eventNotHandledErr;
}

//	===========================================================================
pascal OSStatus CVstWindow::handleHIViewControlEvents(EventHandlerCallRef handler, EventRef theEvent)
{
	if (m_userPane == NULL)
	{
		return eventNotHandledErr;
	}

	OSStatus result = eventNotHandledErr;

	UInt32 myEventClass = GetEventClass(theEvent);
	UInt32 myEventKind  = GetEventKind(theEvent);
	
	switch(myEventClass)
	{
		case kEventClassControl:
			switch(myEventKind)
			{
				case kEventControlDraw:
					{
						//CTrace::trace("Compositing draw called");
						
						// Get the context that we should be using
						CGContextRef theContext;
						GetEventParameter(theEvent, kEventParamCGContextRef, typeCGContextRef,  NULL, sizeof(theContext), NULL, &theContext);
						
						// Get the update region
						RgnHandle regionHandle;
						GetEventParameter(theEvent, kEventParamRgnHandle, typeQDRgnHandle, NULL, sizeof(regionHandle), NULL, &regionHandle);
						
						// And convert it to a rectangle, cos it might not be
						Rect bounds;
						GetRegionBounds(regionHandle, &bounds);
						
						// Lets trace out where it is
						//CTrace::trace("Compositing bounds [%li, %li, %li, %li]", bounds.left, bounds.top, bounds.right - bounds.left, bounds.bottom - bounds.top);
						
						// Rescale the context back to native Quartz format
						CGContextTranslateCTM(theContext, 0, m_windowDimension.getHeight());
						CGContextScaleCTM(theContext, 1.0, -1.0);

						// Save the current state of the context
						CGContextSaveGState(theContext);
	
						// Setup the graphics object
						m_graphics.initialise(m_windowHandle, theContext, m_windowDimension);

						// Store the update area
						CRect myUpdate;
						myUpdate.setFromRect(bounds);
						m_graphics.setUpdateArea(myUpdate);
						m_graphics.resetDrawingArea();

						// Draw the images
						m_theControlRoot->drawRootControl(m_graphics);

						// desetup the graphics
						m_graphics.resetDrawingArea();
						m_graphics.uninitialise();

						// Restore the clipped rectangle
						CGContextRestoreGState(theContext);

						// And we are done
						result = noErr;
					}
				break;
				case kEventControlHitTest:
					CTrace::trace("Hit testing");
					result = noErr;
				break;
				case kEventControlBoundsChanged:
					CTrace::trace("Bounds changed");
				break;
			}
		break;
	}
	
	return result;
}

//	===========================================================================
pascal OSStatus CVstWindow::HIViewControlProc(EventHandlerCallRef handler, EventRef theEvent, void *userData)
{
	if (userData)
	{
		CVstWindow *theWindow = (CVstWindow *)userData;
		return theWindow->handleHIViewControlEvents(handler, theEvent);
	}
	return eventNotHandledErr;
}

#endif

//	===========================================================================
void CVstWindow::idle()
{
	AEffEditor::idle();

#ifndef WIN32
	#ifdef VST_2_4_EXTENSIONS
	if (m_updateAreaIsValid)
	{
		ERect theVSTRect;
		theVSTRect.left   = 0;
		theVSTRect.top    = 0;
		theVSTRect.right  = m_windowDimension.getWidth();
		theVSTRect.bottom = m_windowDimension.getHeight();
		this->draw(&theVSTRect);
	}
	#endif
#endif
}

#if MAC
//	===========================================================================
void CVstWindow::draw (ERect *rect)
{
#ifdef VST_2_4_EXTENSIONS
	if (m_userPane != NULL)
	{
		return;
	}
#endif
	bool deleteRect = false;
	if (rect == NULL)
	{
		rect = new ERect;
		rect->left   = 0;
		rect->top    = 0;
		rect->right  = this->getWindowSize().getWidth(); 
		rect->bottom = this->getWindowSize().getHeight();
		deleteRect  = true;
	}

	// Get the current port
	#ifdef VST_2_4_EXTENSIONS
	CGrafPtr thePort = GetWindowPort(m_windowHandle->m_windowHandle);
	Rect portRect;
	GetPortBounds(thePort, &portRect);
	rect->left   = portRect.left;
	rect->top    = portRect.top;
	rect->right  = portRect.right;
	rect->bottom = portRect.bottom;
	#else
	CGrafPtr thePort;
	GetPort(&thePort);
	#endif
	
	// Setup the qwartz context
	CGContextRef theContext;

	// Start the context drawing
	QDBeginCGContext(thePort, &theContext);

	// Clip to the area
	CGContextSaveGState(theContext);

	// Get the size of the widnow, and position
	Rect windowStructureRgn;
	GetWindowBounds(m_windowHandle->m_windowHandle, kWindowStructureRgn, &windowStructureRgn);
	CDimension dim(abs(windowStructureRgn.left - windowStructureRgn.right), abs(windowStructureRgn.top - windowStructureRgn.bottom));

	// Get the size of the titlebar
	Rect titleBar;
	GetWindowBounds(m_windowHandle->m_windowHandle, kWindowTitleBarRgn, &titleBar);

	// Offset the height by that amount
	dim.setHeight(dim.getHeight() - (titleBar.bottom - titleBar.top));

	// Now initialise the graphics
	m_graphics.initialise(m_windowHandle, theContext, dim);

	// Setup the update region, using a dirty location if we have one
	CRect myUpdate;
	if (m_updateAreaIsValid)
	{
		myUpdate			= m_redrawUpdateArea;
		m_updateAreaIsValid = false;
	}
	else
	{
		myUpdate = CRect(rect->left, rect->top, rect->right - rect->left, rect->bottom - rect->top);
	}
	
	CRect ourArea(CPoint(0, 0), m_windowDimension);
	if (myUpdate.rectIsInside(ourArea))
	{
		myUpdate = ourArea;
	}

	// Store the update area, reset the graphics context
	m_graphics.setUpdateArea(myUpdate);
	m_graphics.resetDrawingArea();

	// Draw the images
	m_theControlRoot->drawRootControl(m_graphics);

	// destup the graphics
	m_graphics.resetDrawingArea();
	m_graphics.uninitialise();

	// Restore the clipped rectangle
	CGContextRestoreGState(theContext);

	// Flush the content to the window
	CGContextFlush(theContext);

	// End the use of the context
	QDEndCGContext(thePort, &theContext);
	
	// Delete as necessary
	if (deleteRect)
	{
		FREE_POINTER(rect);
	}

	// And we are done :)
	return;
}
#endif

#ifndef WIN32
//	===========================================================================
CPoint CVstWindow::getWindowOffset()
{
	return m_theControlRoot->getWindowOffset();
}
#endif

//	===========================================================================
CVstWindow::CParentWindow::CParentWindow(void *windowHandle) : CWindow(CRect(0, 0, 10, 10))
{
#ifdef WIN32
	m_windowHandle->m_windowHandle = (HWND)windowHandle;
#else
	m_windowHandle->m_windowHandle = (WindowRef)windowHandle;
#endif
}

//	===========================================================================
CVstWindow::CParentWindow::~CParentWindow()
{
	NULL_POINTER(m_windowHandle->m_windowHandle);
}

#ifndef WIN32

//	===========================================================================
CVstWindow::CVstWindowTimer::CVstWindowTimer(CVstWindow *window) : m_window(window)
{
	// Nothing to do....
}

//	===========================================================================
CVstWindow::CVstWindowTimer::~CVstWindowTimer()
{
	NULL_POINTER(m_window);		// Cant store as counted, eles mexican standoff
}

//	===========================================================================
void CVstWindow::CVstWindowTimer::timerExpired()
{
	if (m_window)
	{
		m_window->idle();
	}
}
#endif

//#endif